setClass("ebarraysPatterns",
         representation("list",
                        ordered = "logical"))

setClass("ebarraysFamily",
         representation("character",
                        description = "character",
                        link = "function",  ## theta = f(model par)
                        invlink = "function", ## inverse
                        f0 = "function",
                        f0.pp = "function",
                        f0.arglist = "function",
                        deflate = "function",
                        lower.bound = "numeric",
                        upper.bound = "numeric"))

setAs("character", "ebarraysFamily",

      function(from) {

          if (from == "GG") 
              eb.createFamilyGG()
          else if (from == "LNN")
              eb.createFamilyLNN()
          else stop(paste("\n The only families recognized by name are GG (Gamma-Gamma)",
                          "\n and LNN (Lognormal-Normal). Other families need to be",
                          "\n specified as objects of class 'ebarraysFamily'"))
          
      })

setClass("ebarraysEmFit",
         representation(family = "ebarraysFamily",
                        hypotheses = "ebarraysPatterns",
                        thetaEst = "numeric",
                        probEst = "numeric",
                        thetaTrace = "matrix",
                        probTrace = "matrix"))


setMethod("show", "ebarraysEmFit",

          function(object) {

              cat(paste("\n EB model fit",
                        "\n\t Family:", object@family,
                        "(", object@family@description, ")"))
              cat("\n\n Model parameter estimates:\n\n")
              print(object@family@invlink(object@thetaEst))
              cat("\n Estimated mixing proportions:\n\n")
              print(object@probEst)
              cat("\n\n Additional slots: @hypotheses, @thetaTrace, @probTrace\n\n")

              invisible(object)
          })

setClass("HMM.ebarraysPatterns",
         representation(time="numeric",
                        patterns="list",
                        homo="logical",
                        independent="logical"))

HMM.ebPatterns <-
    function(time,x,ordered=F,homo=F,independent=F)
{
    timePts <- sort(unique(time))
    len<-length(timePts)

    if (!is.list(x))
    {
        tmp<-vector("list",1)
        tmp[[1]]<-x
        x<-tmp
    }
    
    if (!(len>1))
        return(ebPatterns(x[[1]]))

    if (len%%length(x)!=0 | len%%length(ordered)!=0)
        stop("Number of time points and number of patterns do not agree")

    patterns <- vector("list", len)
    
    for (i in 1:len)
    {
        patterns[[i]]<-ebPatterns(x[[(i-1)%%length(x)+1]],ordered[[(i-1)%%length(ordered)+1]])
    }
        
    new("HMM.ebarraysPatterns",time=time,patterns=patterns,homo=homo,independent=independent)
}

setClass("HMM.ebarraysEmFit",
         representation(marginal = "list",
                        transit = "list",
                        postProb = "list",
                        path = "matrix",
                        loglik = "list"))

setMethod("show", "HMM.ebarraysEmFit",

          function(object) {
              marginal<-object@marginal
              cat("\n\n Marginal Analysis:\n\n")
              for (i in 1:length(marginal))
              {
                cat(paste("\n\t Time Point",i,":\n\n"))
                cat(paste("\n\t EB model fit",
                          "\n\t\t Family:", marginal[[i]]@family,
                          "\t(", marginal[[i]]@family@description, ")"))
                cat("\n\n\t Model parameter estimates:\n\n")
                print(marginal[[i]]@family@invlink(marginal[[i]]@thetaEst))
                cat("\n\t Estimated mixing proportions:\n\n")
                print(marginal[[i]]@probEst)
                cat("\n\n\t Additional slots: @hypotheses, @thetaTrace, @probTrace\n\n")
              }
              
              cat("\n\n Transition Matrices:\n\n")
              transit<-object@transit
              for (i in 1:length(transit))
              {
                cat(paste("\n From time point",i,"to time point",i+1,":\n\n"))
                tmp<-transit[[i]]
                colnames(tmp)<-paste("Pattern",seq(dim(tmp)[2]))
                rownames(tmp)<-paste("Pattern",seq(dim(tmp)[1]))
                print(tmp)
              }
              
              cat("\n\n Summary of Patterns:\n\n")
              path<-object@path
              for (i in 1:dim(path)[2])
              {
                cat(paste("\n Time point",i,":\n\n"))
                tmp<-table(path[,i])
                names(tmp)<-paste("Pattern",seq(length(tmp)))
                print(tmp)
              }
              
              invisible(object)
          })

output<-function(object,dir="./",pp.path=F) {
              print(paste("Summary information are being stored in ",dir,"summary.txt.", sep=""))
              sink(paste(dir,"summary.txt",sep=""))
              show(object)
              sink()

              print(paste("Most probable paths are being stored in ",dir,"path.txt.", sep=""))
              write.table(object@path,paste(dir,"path.txt", sep=""),quote=F)
              
              print(paste("Marginal posterior probabilities are being stored in ",dir,"postProb.txt.", sep=""))
              postProb<-object@postProb[[1]]
              colNames<-paste("Pattern 1.",seq(dim(postProb)[2]),sep="")
              for (i in 2:length(object@postProb))
              {
                  postProb<-cbind(postProb,object@postProb[[i]])
                  colNames<-c(colNames,paste("Pattern ",i,".",seq(dim(object@postProb[[i]])[2]),sep=""))
              }

              colnames(postProb)<-colNames
              rownames(postProb)<-rownames(object@path)

              write.table(postProb,paste(dir,"postProb.txt", sep=""),quote=F)

              if (pp.path)
              {
                  print(paste("Posterior probabilities of paths are being stored in ",dir,"postprob.path.txt.", sep=""))
                  write.table(probPath(object),paste(dir,"postprob.path.txt", sep=""),quote=F)
              }
              
              invisible(object)
          }

probPath<-function(object) {
              print("This task may take a while, please wait...")
              numPatterns<-1
              for (i in 1:length(object@marginal))
              {
                  numPatterns<-numPatterns*length(object@marginal[[i]]@hypotheses)
              }

              postprob.Path<-matrix(0,dim(object@path)[1],numPatterns)

              pathNames<-NULL

              pat<-rep(0,length(object@marginal))
              
              for (i in 1:numPatterns)
              {
                  denom<-1
                  nomin<-i-1
                  for (j in 1:length(object@marginal))
                  {
                      denom<-length(object@marginal[[j]]@hypotheses)
                      pat[j]<-nomin%%denom+1
                      nomin<-nomin%/%denom

                      if (j==1)
                      {
                          pname<-paste("path.",pat[1],sep="")
                      } else {
                          pname<-paste(pname,"-",pat[j],sep="")
                      }
                  }

                  #print(paste(i,":"))
                  #print(pat)
                  pathNames<-cbind(pathNames,pname)
                  
                  postprob<-object@loglik[[1]][,pat[1]]+log(object@marginal[[1]]@probEst[pat[1]])

                  for (j in 2:length(object@marginal))
                  {
                      postprob<-postprob+log(object@transit[[j-1]][pat[j-1],pat[j]])+object@loglik[[j]][,pat[j]]
                  }

                  postprob.Path[,i]<-exp(postprob)
              }

              postprob.Path<-postprob.Path/drop(apply(postprob.Path,1,sum))
              colnames(postprob.Path)<-pathNames
              rownames(postprob.Path)<-rownames(object@path)

              return(postprob.Path)
          }
    
setGeneric("emfit",
           function(data, family, hypotheses, theta.init, p.init, ...) standardGeneric("emfit"))


setMethod("emfit",
          signature(data = "exprSet",
                    family = "character",
                    hypotheses = "HMM.ebarraysPatterns",
                    theta.init = "vector",
                    p.init = "vector"),

          function(data,
                   family,
                   hypotheses,
                   theta.init,
                   p.init, ...) {

              data <- exprs(data)
              family <- as(family, "ebarraysFamily")
              callGeneric(data = data,
                          family = family,
                          hypotheses = hypotheses,
                          theta.init = theta.init,
                          p.init = p.init, ...)

          })



setMethod("emfit",
          signature(data = "matrix",
                    family = "character",
                    hypotheses = "HMM.ebarraysPatterns",
                    theta.init = "vector",
                    p.init = "vector"),

          function(data,
                   family,
                   hypotheses,
                   theta.init,
                   p.init, ...) {

              family <- as(family, "ebarraysFamily")
              callGeneric(data = data,
                          family = family,
                          hypotheses = hypotheses,
                          theta.init = theta.init,
                          p.init = p.init, ...)

          })



setMethod("emfit",
          signature(data = "exprSet",
                    family = "ebarraysFamily",
                    hypotheses = "HMM.ebarraysPatterns",
                    theta.init = "vector",
                    p.init = "vector"),

          function(data,
                   family,
                   hypotheses,
                   theta.init,
                   p.init, ...) {

              data <- exprs(data)
              callGeneric(data = data,
                          family = family,
                          hypotheses = hypotheses,
                          theta.init = theta.init,
                          p.init = p.init, ...)

          })


setMethod("emfit",
          signature(data = "matrix",
                    family = "ebarraysFamily",
                    hypotheses = "HMM.ebarraysPatterns",
                    theta.init = "vector",
                    p.init = "vector"),

          function(data,
                   family,
                   hypotheses,
                   theta.init,
                   p.init, ...) {

              timePts<-unique(hypotheses@time)
              len<-length(timePts)
              family.all<-vector("list",len)
              p.init.all<-vector("list",len)
              theta.init.all<-vector("list",len)
              for (i in 1:len)
              {
                p.init.all[[i]]<-p.init
                theta.init.all[[i]]<-theta.init
                family.all[[i]]<-family
              }
              
              callGeneric(data = data,
                          family = family.all,
                          hypotheses = hypotheses,
                          p.init = p.init.all,
                          theta.init = theta.init.all, ...)

          })

setMethod("emfit",
          signature(data = "matrix",
                    family = "list",
                    hypotheses = "HMM.ebarraysPatterns",
                    theta.init = "list",
                    p.init = "list"),

          function(data,
                   family,
                   hypotheses,
                   theta.init, p.init,
                   num.iter = 20,
                   verbose = interactive(),
                   trace = TRUE,
                   optim.control = list(),
                   ...) {

              ## Store current process time
              if (exists("proc.time")) startTime <- proc.time()

              ## Define function to be optimized by optim:

              complete.loglik <-
                  function(theta,
                           zhat,
                           zhatlogp,
                           hypotheses,
                           f0, f0.args)
                  {
                      pred.dens <- zhat # values unimportant, just need the same dimensions

                      ## NOTE: f0 returns the log value of the predictive probability
                      ## density function.
                      for (i in seq(along = hypotheses))
                      {
                          foo <- 0 #numeric(nrow(zhat))
                          for (j in seq(along = hypotheses[[i]]))
                          {
                              foo <- 
                                  foo +
                                      f0(theta = theta,
                                         args = c(f0.args$common.args,
                                         f0.args$pattern.args[[i]][[j]]))
                          }
                          pred.dens[, i] <- foo
                      }
                      -(sum(zhat * pred.dens) + zhatlogp)
                  }


              ## Some preprocessing:

              verbose <- as.numeric(verbose)

              if (verbose > 0) cat(paste("\n Checking for negative entries..."))
              all.pos <- apply(data, 1, function(x) !any(x <= 0))
              if (verbose > 0 && sum(all.pos) != length(all.pos)) {
                  cat(paste("\n\t", length(all.pos) - sum(all.pos),
                            "rows out of", length(all.pos),
                            "had at least one negative entry"))
                  cat("\n\t These rows will not be used in the EM fit\n")
              }


              if (verbose > 0)
                  cat(paste("\n Generating summary statistics for patterns.",
                            "\n This may take a few seconds...\n"))

              timePts<-sort(unique(hypotheses@time))
              len<-length(timePts)
              dataObj<-vector("list",len)
              patternObj<-vector("list",len)
              hypothesesObj<-vector("list",len)
              f0.args<-vector("list",len)
              theta.temp<-vector("list",len)
              theta.mat<-vector("list",len)
              p.temp<-vector("list",len)
              p.mat<-vector("list",len)
              forward<-vector("list",len)
              backward<-vector("list",len)
              postprob<-vector("list",len)
              ordered<-rep(T,len)

              for (i in 1:len)
              {
                dataObj[[i]]<-data[,hypotheses@time==timePts[i]]
                rownames(dataObj[[i]])<-rownames(data)
                patternObj[[i]] <- hypotheses@patterns[[i]]
                hypothesesObj[[i]] <- as(hypotheses@patterns[[i]], "list")
                ordered[i]<-hypotheses@patterns[[i]]@ordered
                
                f0.args[[i]] <-
                  family[[i]]@f0.arglist(dataObj[[i]][all.pos, ],
                                    hypothesesObj[[i]])

                theta.temp[[i]] <- family[[i]]@link(theta.init[[i]])
                theta.mat[[i]] <- matrix(NA, num.iter + 1, length(theta.temp[[i]]))
                theta.mat[[i]][1,] <- theta.temp[[i]]

                p.init[[i]] <- p.init[[i]] / sum(p.init[[i]])
                p.temp[[i]] <- p.init[[i]]
                p.mat[[i]] <- matrix(NA, num.iter + 1, length(p.init[[i]]))
                p.mat[[i]][1,] <- p.init[[i]]

                if (length(p.init[[i]])!=length(hypothesesObj[[i]]))
                    stop(paste("the number of patterns does not match the number of initial probabilities at time", i))
              }

              rm(data)
              
              transit.init<-vector("list",(len-1))
              transit.temp<-vector("list",(len-1))
              for (i in 1:(len-1)) 
              {
                transit.init[[i]]<-matrix(1/length(hypothesesObj[[i]]),length(hypothesesObj[[i]]),length(hypothesesObj[[i+1]]))
                transit.temp[[i]]<-transit.init[[i]]*0
              }

              transit.pre<-transit.init

              ## Start Iterations
              if (verbose > 0)
                  cat(paste("\n Starting EM iterations (total", num.iter,
                            ").\n This may take a while\n\n"))

              notdone <- TRUE ## not used now, perhaps some convergence criteria
              iter <- 0
              zhat<-vector("list",len)
              for (i in 1:len)
              {
                zhat[[i]] <- matrix(0, sum(all.pos), length(hypothesesObj[[i]]))
              }
              
              while ({iter <- iter + 1;  (iter <= num.iter && notdone)})
              {

                  if (verbose > 0)
                      cat(paste("\t Starting iteration", iter, "...\n"))
                  
                  ## E step
                  for (i in 1:len)
                  {
                    zhat[[i]][,] <- 0
                  }
                  
                  ## NOTE: f0 returns the log value of the predictive
                  ## probability density function. Thus to get the log of the
                  ## product of the predictive probability densities, we add
                  ## (log(PA=1 * PA=2) = log(PA=1) + log(PA=2)).

                  for (k in 1:len)
                  {
                    for (i in seq(along = hypothesesObj[[k]]))
                    {
                      foo <- 0
                      for (j in seq(along = hypothesesObj[[k]][[i]]))
                      {
                          foo <- 
                              foo + 
                                  family[[k]]@f0(theta = theta.temp[[k]],
                                            args = c(f0.args[[k]]$common.args,
                                            f0.args[[k]]$pattern.args[[i]][[j]]))
                      }
                      if (ordered[k] & length(hypothesesObj[[k]][[i]])>1)
                      {
                        zhat[[k]][, i] <- foo+family[[k]]@deflate(theta = theta.temp[[k]],f0.args[[k]]$pattern.args[[i]])
                      } else {
                        zhat[[k]][, i] <- foo
                      }
                    }

                    #zhat[[k]] <- .Call("makeProbVect", zhat[[k]], p.temp[[k]], PACKAGE = "EBarrays")

                    #p.temp[[k]] <- colSums(zhat[[k]])
                    #p.temp[[k]] <- p.temp[[k]] / sum(p.temp[[k]])
                    #if (verbose > 1) {
                    #  cat("\n\n      p.temp: ")
                    #}
                  }
                  
                  #forward
                  for (i in 1:len) 
                  {
                    if (i==1)
                    {
                      forward[[i]]<-t(p.temp[[i]]*t(exp(zhat[[i]])))
                    } else {
                      forward[[i]]<-(forward[[i-1]]%*%transit.pre[[i-1]])*exp(zhat[[i]])
                    }
                    forward[[i]]<-forward[[i]]/drop(apply(forward[[i]],1,sum))
                  }

                  #backward
                  for (i in len:1) 
                  {
                    if (i==len) 
                    {
                      backward[[i]]<-matrix(1, sum(all.pos), length(hypothesesObj[[i]]))
                    } else {
                      backward[[i]]<-exp(zhat[[i+1]]+log(backward[[i+1]]))%*%t(transit.pre[[i]])
                    }
                    backward[[i]]<-backward[[i]]/drop(apply(backward[[i]],1,sum))
                  }

                  #update posterior prob and transit
                  for (i in 1:len) 
                  {
                    postprob[[i]]<-exp(log(forward[[i]])+log(backward[[i]]))
                    postprob[[i]]<-postprob[[i]]/drop(apply(postprob[[i]],1,sum))
                  }

                  for (i in 1:len) 
                  {
                    p.temp[[i]]<-drop(apply(postprob[[i]],2,sum))
                    p.temp[[i]]<-p.temp[[i]]/sum(p.temp[[i]])
                  }

                  for (i in 1:(len-1)) 
                  {
                    transit.temp[[i]][,]<-0
                  }

                  for (i in 2:len) 
                  {
                    if (hypotheses@independent)
                    {
                      for (j in 1:length(hypothesesObj[[i-1]]))
                      {
                        transit.temp[[i-1]][j,]<-p.temp[[i]]
                      }
                    } else {
                      for (pos in 1:sum(all.pos)) 
                      {
                        transit.curr<-((forward[[i-1]][pos,])%*%t((backward[[i]][pos,])*(exp(zhat[[i]][pos,]))))*transit.pre[[i-1]]
                        transit.curr<-transit.curr/sum(transit.curr)
                        transit.temp[[i-1]]<-transit.temp[[i-1]]+transit.curr
                      }
                    }
                  }

                  if (hypotheses@homo) 
                  {
                    transit.curr<-transit.temp[[1]]*0
                    
                    for (i in 1:(len-1)) 
                    {
                      transit.curr<-transit.curr+transit.temp[[i]]
                    }
                    for (i in 1:(len-1)) 
                    {
                      transit.temp[[i]]<-transit.curr
                    }
                 }

                 for (i in 1:(len-1)) 
                 { 
                   for (j in 1:length(hypothesesObj[[i]])) 
                   {
                     transit.temp[[i]][j,]<-transit.temp[[i]][j,]/(sum(transit.temp[[i]][j,]))
                   }
                 }
    
                 transit.pre<-transit.temp
                  
                  ## M step
                  for (i in 1:len)
                  {
                    ans<-matrix(0,sum(all.pos),length(hypothesesObj[[i]]))
                    
                    for (ii in 1:sum(all.pos))
                    {
                      for (j in 1:length(hypothesesObj[[i]]))
                      {
                        ans[ii,j] = 0;
                        for (k in 1:length(hypothesesObj[[i]]))
                          ans[ii,j] = ans[ii,j]+exp(zhat[[i]][ii,k] + log(p.temp[[i]][k]) - zhat[[i]][ii,j] - log(p.temp[[i]][j]));
                        ans[ii,j] = 1 / ans[ii,j];
                      }
                    }
                  
                    #zhat <- .Call("makeProbVect", zhat, p.temp, PACKAGE = "EBarrays")
                    zhatlogp <- sum(ans %*% log(p.temp[[i]]))
                    theta.temp[[i]] <-
                      optim(par = theta.temp[[i]],
                            fn = complete.loglik,
                            method = c("L-BFGS-B"),
                            lower = family[[i]]@lower.bound,
                            upper = family[[i]]@upper.bound,
                            control = optim.control,
                            zhat = ans,
                            zhatlogp = sum(ans %*% log(p.temp[[i]])),
                            hypotheses = hypothesesObj[[i]],
                            f0 = family[[i]]@f0,
                            f0.args = f0.args[[i]])$par

                    theta.mat[[i]][iter + 1,] <- theta.temp[[i]]
                    p.mat[[i]][iter + 1,] <- p.temp[[i]]

                    if (verbose > 1) {
                        cat("\n\n  theta.temp: ")
                        print(theta.temp[[i]])
                    }
                  }
              }

              ## Finish up

              for (i in 1:len)
              {
                colnames(theta.mat[[i]]) <- names(theta.temp[[i]]) <-
                  c(paste("theta", 1:(length(theta.temp[[i]])), sep=""))
                colnames(p.mat[[i]]) <- names(p.temp[[i]]) <-
                  c(paste("p", 1:(length(p.temp[[i]])), sep=""))
              }
              
              if (verbose > 1) {
                  cat("\n\n")
                  print("model parameter estimates:")
                  print(theta.temp)
                  print("Estimates of mixing proportions:")
                  print(p.temp)
              }

              if (verbose && exists("proc.time")) {
                  endTime <- proc.time()
                  cat(sprintf("\n\n Fit used %5.2f seconds user time\n",
                              endTime[1] - startTime[1]))
              }

              out1<-vector("list",len)
              for (i in 1:len)
              {
                if (trace)
                {
                  out1[[i]]<-new("ebarraysEmFit", 
                      family = family[[i]],
                      hypotheses = patternObj[[i]],
                      thetaEst = theta.temp[[i]],
                      probEst = p.temp[[i]],
                      thetaTrace = theta.mat[[i]],
                      probTrace = p.mat[[i]])
                } else {
                  out1[[i]]<-new("ebarraysEmFit", 
                      family = family[[i]],
                      hypotheses = patternObj[[i]],
                      thetaEst = theta.temp[[i]],
                      probEst = p.temp[[i]])
                }
             }
             
             #viterbi
             paths<-matrix(NA,sum(all.pos),len)
             cur.path<-rep(NA,len)
             delta<-vector("list",len)
             psi<-vector("list",len)
             for (i in 1:len)
             {
               delta[[i]]<-rep(NA,length(hypothesesObj[[i]]))
               psi[[i]]<-rep(NA,length(hypothesesObj[[i]]))
               #print("1")
             }
             
             for (i in 1:sum(all.pos)) {
               psi[[1]]<-rep(NA,length(hypothesesObj[[1]]))
               delta[[1]]<-(zhat[[1]][i,]+log(p.temp[[1]]))
               #print("2")

               for (j in 2:len) 
               {
                 tmp<-delta[[j-1]]+log(transit.temp[[j-1]])
                 for (k in 1:length(hypothesesObj[[j]])) 
                 {
                   psi[[j]][k]<-order(-tmp[,k])[1]
                   delta[[j]][k]<-max(tmp[,k])
                   #print("3")
                 }
                 delta[[j]]<-delta[[j]]+zhat[[j]][i,]
               }

               cur.path[len]<-order(-delta[[len]])[1]

               for (j in len:2) {
                 cur.path[j-1]<-psi[[j]][cur.path[j]]
                 #print("4")
               }

               paths[i,]<-cur.path
             }
             
             colnames(paths)<-paste("Time",1:len)
             rownames(paths)<-rownames(dataObj[[1]])[all.pos]
             
             new("HMM.ebarraysEmFit",marginal=out1,transit=transit.temp,postProb=postprob,path=paths,loglik=zhat)
          })


ebPatterns <-
    function(x,ordered=F)
    ## x can be a character vector (of length > 2), a connection or a
    ## file name
{
    if (!(is.character(x) && length(x) > 1))
        x <- readLines(x)
    patterns <- vector("list", length(x))
    len <- FALSE
    for(i in seq(along = patterns))
    {
        pat <- as.numeric(strsplit(x[i], "\\W+")[[1]])
        if (is.logical(len))
        {
            len <- length(pat)
            if (len == 0) stop("Pattern has length 0")
        }
        if (length(pat) != len || any(is.na(pat)))
        {
            print(pat)
            stop("Invalid pattern")
        }
        vals <- sort(unique(pat[pat > 0]))
        patterns[[i]] <- vector("list", length(vals))
        for (j in seq(along = vals))
        {
            patterns[[i]][[j]] <- (1:len)[pat == vals[j]]
        }
    }
    new("ebarraysPatterns", patterns, ordered=ordered)
}

eb.createFamilyGG <-
    function()
{

    f0 <- function(theta, args)
    {
        ## alpha=theta[1], alpha0=theta[2], nu=theta[3]
        theta[2] * log(theta[3]) +
            lgamma(args$n * theta[1] + theta[2]) -
                args$n * lgamma(theta[1]) - lgamma(theta[2]) + 
                    ( (theta[1] - 1) * args$lprod.data) -
                        ( (args$n * theta[1] + theta[2]) * log(args$sum.data + theta[3])  )
    }


    f0.pp <- function(theta, args)
    {
        ## alpha=theta[1], alpha0=theta[2], nu=theta[3]
        ll1 <- theta[2] * log(theta[3]) +
            lgamma(args$n * theta[1] + theta[2]) -
                args$n * lgamma(theta[1]) - lgamma(theta[2])

        ll2 <- - (args$n * theta[1] + theta[2]) * (log(args$sum.data + theta[3]))

        ll1 + ll2
    }

    f0.arglist <- function(data, patterns)
    {

        ## returns a list with two components, common.args and
        ## pattern.args. common.args is a list of arguments to f0 that
        ## don't change from one pattern to another (not used
        ## currently for speed), whereas pattern.args[[i]][[j]] is a
        ## similar list of arguments, but specific to the columns in
        ## pattern[[i]][[j]]

        ## Note: f0 will also have an argument theta, which would need
        ## to be specified separately

        common.args <- list() ## nothing for GG
        pattern.args <- vector("list", length(patterns))
        for (i in seq(along = patterns))
        {
            pattern.args[[i]] <- vector("list", length(patterns[[i]]))
            for (j in seq(along = patterns[[i]]))
            {
                tmpdata <- data[, patterns[[i]][[j]], drop = FALSE]
                nn <- length(patterns[[i]][[j]])
                lprod.data <- rowSums(log(tmpdata))
                sum.data <- rowSums(tmpdata)
                pattern.args[[i]][[j]] <-
                    list(n = nn, lprod.data = lprod.data,
                         sum.data = sum.data)
            }
        }
        list(common.args = common.args, pattern.args = pattern.args)
    }

    deflate <- function(theta, args, num.sample=10000)
    {
        ## alpha=theta[1], alpha0=theta[2], nu=theta[3]

        num.groups<-length(args)
	a <- rep(0,num.groups)
	b <- matrix(0,length(args[[1]]$sum.data),num.groups)
	
	for (i in 1:num.groups)
	{
          a[i]<-args[[i]]$n*theta[1]+theta[2]
          b[,i]<-args[[i]]$sum.data+theta[3]
        }
        
        if (num.groups==2)
        {
          ans <- pbeta(b[,1]/(b[,1]+b[,2]),a[1],a[2])
        } else {
          V.sample<-matrix(0,num.sample,num.groups)
          for (i in 1:num.groups)
          {
            V.sample[,i]<-rgamma(num.sample,shape=a[i],rate=1)
          }

          ans <- rep(0,length(args[[i]]$sum.data))
          
          for (i in 1:length(args[[i]]$sum.data))
          {
            ans.vec<-rep(T,num.sample)
            for (j in 1:(num.groups-1))
            {
              ans.vec<-ans.vec & (V.sample[,j]/b[i,j]>V.sample[,j+1]/b[i,j+1])
            }
            ans[i]<-mean(ans.vec)
          }
        }
        return(log(gamma(num.groups+1)*ans))
    }
    
    new("ebarraysFamily",
        "GG",
        description = "Gamma-Gamma",
        link = function(x) x, ## theta = (alpha, alpha_0, nu)
        invlink = function (x) {
            names(x) <- c("alpha", "alpha0", "nu")
            x
        },
        f0 = f0,
        f0.pp = f0.pp,
        f0.arglist = f0.arglist,
        deflate = deflate,
        lower.bound  = c(1.01, 0.01, 0.01),
        upper.bound = c(10000, 10000, 10000))
}

eb.createFamilyLNN <-
    function()
{

    f0 <- function(theta, args)
    {
        ## mu0=theta[1], log(sigma2)=theta[2], log(tau02)=theta[3]

        theta[2:3] <- exp(theta[2:3])
        spnt <- theta[2] + args$n * theta[3] ## sigma^2 + n tao0^2
        negloglik <-

            c(args$n, args$n - 1, 1) %*%
                log(c(2 * base::pi, theta[2], spnt)) +
                    
                    (args$n * theta[1] * theta[1] + args$sumsq.logdata - 2
                     * theta[1] * args$sum.logdata) / theta[2] -
                         
                         theta[3] * (args$sum.logdata * args$sum.logdata +
                                     args$n * args$n * theta[1] * theta[1]
                                     - 2 * args$n * theta[1] *
                                     args$sum.logdata ) / (theta[2] *
                                                           spnt)

        ##print(str(negloglik))
        -negloglik
    }

    f0.arglist <- function(data, patterns)
    {

        ## returns a list with two components, common.args and
        ## pattern.args. common.args is a list of arguments to f0 that
        ## don't change from one pattern to another, whereas
        ## pattern.args[[i]][[j]] is a similar list of arguments, but
        ## specific to the columns in pattern[[i]][[j]]

        ## Note: f0 will also have an argument theta, which would need
        ## to be specified separately

        data <- log(data)
        k <- nrow(data)
        common.args <- list() ## nothing for LNN
        pattern.args <- vector("list", length(patterns))
        for (i in seq(along = patterns))
        {
            pattern.args[[i]] <- vector("list", length(patterns[[i]]))
            for (j in seq(along = patterns[[i]]))
            {
                tmpdata <- data[, patterns[[i]][[j]], drop = FALSE]
                pattern.args[[i]][[j]] <-
                    list(n = length(patterns[[i]][[j]]),
                         sum.logdata = rowSums(tmpdata),
                         sumsq.logdata = rowSums(tmpdata * tmpdata))
            }
        }
        list(common.args = common.args, pattern.args = pattern.args)
    }

    deflate <- function(theta, args, num.sample=10000)
    {
        ## mu0=theta[1], log(sigma2)=theta[2], log(tau02)=theta[3]

        theta[2:3] <- exp(theta[2:3])
        num.groups<-length(args)
	spnt <- rep(0,num.groups)
	a <- matrix(0,length(args[[1]]$sum.logdata),num.groups)
	b <- rep(0,num.groups)
	
	for (i in 1:num.groups)
	{
          spnt[i] <- theta[2] + args[[i]]$n * theta[3] ## sigma^2 + n tao0^2        
          a[,i]<-(theta[2]*theta[1] + theta[3]*args[[i]]$sum.logdata)/spnt[i]
          b[i]<-theta[2] * theta[3]/spnt[i]
        }
        
        if (num.groups==2)
        {
          ans <- pnorm((a[,1]-a[,2])/sqrt(b[1]^2+b[2]^2))
        } else {
          V.sample<-matrix(rnorm(num.sample*num.groups),num.sample,num.groups)
          for (i in 1:num.groups)
          {
            V.sample[,i]<-V.sample[,i]*b[i]
          }

          ans <- rep(0,length(args[[i]]$sum.logdata))
          
          for (i in 1:length(args[[i]]$sum.logdata))
          {
            ans.vec<-rep(T,num.sample)
            for (j in 1:(num.groups-1))
            {
              ans.vec<-ans.vec & (V.sample[,j]+a[i,j]>V.sample[,j+1]+a[i,j+1])
            }
            ans[i]<-mean(ans.vec)
          }
        }
        return(log(gamma(num.groups+1)*ans))
    }

    new("ebarraysFamily",
        "LNN",
        description = "Lognormal-Normal",
        link = function(x) {
            x[2:3] <- log(x[2:3])
            ## theta = (mu_0, log(sigma^2), log(tao_0^2))
            x 
        },
        invlink = function (x) {
            x[2:3] <- exp(x[2:3])
            names(x) <- c("mu_0", "sigma^2", "tao_0^2")
            x
        },
        f0 = f0,
        f0.pp = f0,
        f0.arglist = f0.arglist,
        deflate = deflate,
        lower.bound = c(-Inf, -Inf, -Inf), 
        upper.bound = c(Inf, Inf, Inf))
}
